/* psudoku.cpp: functions to create/solve Sudoku-like puzzles
and print them out as PostScript files

Copyright (C) 2013, Project Pluto

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.    */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdint.h>
#ifdef _MSC_VER
#include <conio.h>
#endif

typedef uint64_t poss_t;
// typedef unsigned long poss_t;

int verbose = 0;
int n_groups;
poss_t all_possibles;

#define SUDOKU struct sudoku
#define MAX_GRID_SIZE (sizeof( poss_t) * 8 - 1)
#define MAX_N_GROUPS (MAX_GRID_SIZE * 3 + 2)

#define PUZZLE_IS_UNSOLVABLE -2
#define MULTIPLE_SOLUTIONS_FOUND -3

/* For each Sudoku,  we store the 'solutions' (0 if that cell is still
an unknown,  otherwise 1-9 for the usual 9x9 grid),  and the 'possible'
values for that cell (0 if the cell is solved for;  otherwise,  a bit
mask of possible values).  For example,  a 'possible' value of 0x18 means
that cell could contain either a three or a four.  If we somehow ruled
out the three,  then the 'possible' value would change to 0x10... and
at some point,  the 'look_for_single_candidates()' function will notice
that the 'possible' value is a power of two,  therefore only one bit is
set,  therefore only one number is possible in this cell and it can be
considered solved. */


SUDOKU
   {
   int grid_size, n_cells;
   char *solutions;
   poss_t *possibles;
   };

#ifdef NDEBUG
#define check_sudoku( X)
#else
static void check_sudoku( const SUDOKU *su)
{
   assert( su->solutions[0] >= 0);
   assert( su->solutions[0] <= su->grid_size);
}
#endif

static SUDOKU *generate_blank_sudoku( const int grid_size)
{
   const int n_cells = grid_size * grid_size;
   SUDOKU *rval = (SUDOKU *)calloc( 1, sizeof( SUDOKU)
               + n_cells * (1 + sizeof( poss_t)));

   rval->solutions = (char *)( rval + 1);
   rval->possibles = (poss_t *)( rval->solutions + n_cells);
   rval->grid_size = grid_size;
   rval->n_cells = n_cells;
   check_sudoku( rval);
   return( rval);
}

      /* When copying sudoku,  copy just the 'solutions' and 'possibles'
         data trailing the SUDOKU structure.  _Don't_ copy the other data. */
static void copy_sudoku( SUDOKU *dest, const SUDOKU *src)
{
   memcpy( dest->solutions, src->solutions, src->n_cells * sizeof( char));
   memcpy( dest->possibles, src->possibles, src->n_cells * sizeof( poss_t));
}

/* As the following macro indicates,  it's possible to swap integer */
/* values without need for an intermediate value:                   */
#define SWAP_INTEGER_VALUES( x, y)  { x ^= y; y ^= x; x ^= y; }

/* The 'groups' array indicates which cells belong to which group.
(A group is a row,  column,  or 3x3 block,  or some other sort of
block for a non-3x3-standard Sudoku.  That is,  a "group" is a collection
of cells in which each number is to occur exactly once in the solution.)
Sometimes,  the diagonals are also groups. */

static int groups[MAX_GRID_SIZE * MAX_N_GROUPS];

/* These group names don't help with anything but 9x9 grids... */

static const char *group_names[29] = {
      "upper left block",
      "upper middle block",
      "upper right block",
      "middle left block",
      "center block",
      "middle right block",
      "lower left block",
      "lower middle block",
      "lower right block",
      "top row",
      "second row",
      "third row",
      "fourth row",
      "middle row",
      "sixth row",
      "seventh row",
      "eighth row",
      "bottom row",
      "left column",
      "second column",
      "third column",
      "fourth column",
      "middle column",
      "sixth column",
      "seventh column",
      "eighth column",
      "right column",
      "descending diagonal",
      "ascending diagonal" };

/* The 'block_membership' array gives the block to which each cell
corresponds.  It's set up in 'load_grid_data()'. */

char *block_membership;

void set_groups( int *groups, const char *block_membership, const int grid_size)
{
   int i, j;

               /* first,  find blocks: */
   for( i = 0; i < grid_size; i++)
      for( j = 0; j < grid_size * grid_size; j++)
         if( block_membership[j] == i)
            *groups++ = j;
               /* next,  find rows... */
   for( i = 0; i < grid_size; i++)
      for( j = 0; j < grid_size; j++)
         *groups++ = i * grid_size + j;
               /* ...then invert to find columns: */
   for( i = 0; i < grid_size; i++)
      for( j = 0; j < grid_size; j++)
         *groups++ = j * grid_size + i;
               /* ...then add one diagonal,  then the other.  These
                  will only actually get used with the '-x' switch: */
   for( i = 0; i < grid_size; i++)
      *groups++ = i * (grid_size + 1);
   for( i = 0; i < grid_size; i++)
      *groups++ = (i + 1) * (grid_size - 1);
}

int solve_sudoku( SUDOKU *su);

static int count_bits( poss_t ival)
{
   int rval = 0;

   while( ival)
      {
      if( ival & 1)
         rval++;
      ival >>= 1;
      }
   return( rval);
}

#ifdef NON_STRAIGHTFORWARD_WAY
/* Less straightforward way to count the bits in a 32-bit int... */
/* Needs a bit of modification for 64-bit ints anyway */
static int count_bits( unsigned ival)
{
   ival -= ((ival >> 1) & 0x55555555);
   ival = (((ival >> 2) & 0x33333333) + (ival & 0x33333333));
   ival = (((ival >> 4) + ival) & 0x0f0f0f0f);
   ival += (ival >> 8);
   ival += (ival >> 16);
   return( ival & 0x3f);
}
#endif

int set_possibles( SUDOKU *su, const int idx)
{
   const poss_t mask = ~((poss_t)1 << su->solutions[idx]);
   int i, j, group, idx2 = 0;

   su->possibles[idx] = 0;
   for( group = 0; group < n_groups; group++)
      for( i = 0; i < su->grid_size; i++, idx2++)
         if( groups[idx2] == idx)
            for( j = 0; j < su->grid_size; j++)
               su->possibles[groups[j + group * su->grid_size]] &= mask;
   return( 0);
}

void set_all_possibles( SUDOKU *su)
{
   int i, group;

   for( i = 0; i < su->n_cells; i++)
      su->possibles[i] = (su->solutions[i] ? (poss_t)0 : all_possibles);
// for( i = 0; i < su->n_cells; i++)
//    if( su->solutions[i])
//       set_possibles( su, i);
   for( group = 0; group < n_groups; group++)
      {
      const int *group_ptr = groups + group * su->grid_size;
      poss_t mask = all_possibles;

      for( i = 0; i < su->grid_size; i++)
         if( su->solutions[group_ptr[i]])
            mask ^= ((poss_t)1 << su->solutions[group_ptr[i]]);
      for( i = 0; i < su->grid_size; i++)
         if( !su->solutions[group_ptr[i]])
            su->possibles[group_ptr[i]] &= mask;
      }
}

/* Returns PUZZLE_IS_UNSOLVABLE if no solution is possible (because a
particular number can't be fitted into a particular group);  -1 if
no singleton was found; or the index of the cell (0-80) wherein a
singleton was found. */

int look_for_singletons_group( SUDOKU *su, const int *group_idx)
{
   int i, rval = -1;
   poss_t already_found = 0;

   check_sudoku( su);
   assert( su->solutions[0] >= 0);
   assert( su->solutions[0] <= su->grid_size);
            /* make a bit-mask of the numbers already found in this group */
   for( i = 0; i < su->grid_size; i++)
      already_found |= ((poss_t)1 << su->solutions[group_idx[i]]);
            /* then,  for all numbers not already found in the group... */
   for( i = 1; i <= su->grid_size; i++)
      if( !((already_found >> i) & 1))
         {
         int n_found = 0, j;
         const poss_t mask = ((poss_t)1 << i);

                     /* see in how many cells that number might be placed */
         for( j = 0; j < su->grid_size && n_found < 2; j++)
            {
            const int idx = group_idx[j];

            if( su->possibles[idx] & mask)
               {
               rval = idx;
               n_found++;
               }
            }
                     /* if that number could only go in one place, */
         if( n_found == 1)     /* we have our singleton */
            {
            su->solutions[rval] = (char)i;
            set_possibles( su, rval);
            return( rval);
            }
         if( !n_found)      /* but if it couldn't go in _any_ place in */
            return( PUZZLE_IS_UNSOLVABLE); /* this group,  it's insoluble */
         }
   return( -1);
}

/* look_for_singletons( ) either returns the number 1-9 of a cell that was
found;  or -1,  meaning 'no singletons found';  or PUZZLE_IS_UNSOLVABLE,
meaning 'no solution is possible for this puzzle'. */

int look_for_singletons( SUDOKU *su, int *group_idx)
{
   int group, cell;

   for( group = 0; group < n_groups; group++)
      {
      cell = look_for_singletons_group( su, groups + group * su->grid_size);
      if( cell >= 0 || cell == PUZZLE_IS_UNSOLVABLE)
         {
         *group_idx = group;
         return( cell);
         }
      }
   return( -1);
}

/* look_for_single_candidates( ) returns the cell index (0-80) wherein */
/* a single candidate was found;  or -1 if no single candidate is found. */

#define IS_POWER_OF_TWO( x)  (!(x & (x - 1)))

int look_for_single_candidates( SUDOKU *su)
{
   int i, j;

   for( i = 0; i < su->n_cells; i++)
      if( !su->solutions[i])
         if( IS_POWER_OF_TWO( su->possibles[i]))
            for( j = 1; j <= su->grid_size; j++)
               if( su->possibles[i] == ((poss_t)1 << j))
                  {
                  su->solutions[i] = (char)j;
                  set_possibles( su, i);
                  return( i);
                  }
   return( -1);
}

int look_for_candidate_lines_in_block( SUDOKU *su, const int block_number)
{
   int i, search_num, rval = -1;
   int *tptr = groups + block_number * su->grid_size;

   for( search_num = 1; search_num <= su->grid_size; search_num++)
      {
      int row0 = -1, row, col0 = -1, col;
      const poss_t mask = ((poss_t)1 << search_num);

      for( i = 0; i < su->grid_size; i++)
         {
         int loc = tptr[i];

         if( su->possibles[loc] & mask)
            {
            row = loc / su->grid_size;
            if( row0 == -1)
               row0 = row;
            else if( row != row0)  /* possibilities exist in _two_ rows */
               row0 = -2;          /* so it's a failure */

            col = loc % su->grid_size;
            if( col0 == -1)
               col0 = col;
            else if( col != col0)  /* possibilities exist in _two_ cols */
               col0 = -2;          /* so it's a failure */
            }
         }

      if( row0 > -1)     /* only candidates for search_num are in one row */
         {
         int loc = row0 * su->grid_size;

         for( i = 0; i < su->grid_size; i++, loc++)
            if( block_membership[loc] != block_number)
               if( su->possibles[loc] & mask)
                  {
                  su->possibles[loc] ^= mask;
                  rval = search_num;
                  }
         }

      if( rval >= 0)
         return( rval);

      if( col0 > -1)     /* only candidates for search_num are in one col */
         {
         int loc = col0;

         for( i = 0; i < su->grid_size; i++, loc += su->grid_size)
            if( block_membership[loc] != block_number)
               if( su->possibles[loc] & mask)
                  {
                  su->possibles[loc] ^= mask;
                  rval = search_num;
                  }
         }

      if( rval >= 0)
         return( rval);
      }
   return( -1);
}

int look_for_candidate_lines( SUDOKU *su, int *group_idx)
{
   int block_number, rval = -1;

   for( block_number = 0; block_number < su->grid_size; block_number++)
      {
      rval = look_for_candidate_lines_in_block( su, block_number);
      if( rval >= 0)
         {
         *group_idx = block_number;
         return( rval);
         }
      }
   return( rval);
}

int look_for_hidden_pairs_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, rval = -1;
   poss_t total_mask = 0;

   for( i = 0; i < su->grid_size; i++)
      total_mask |= su->possibles[group_ptr[i]];
   for( i = 2; i <= su->grid_size; i++)
      if( (total_mask >> i) & 1)
         for( j = 1; j < i; j++)
            if( (total_mask >> j) & 1)
               {
               const poss_t mask = (((poss_t)1 << i) | ((poss_t)1 << j));
               int n_with_pair = 0, idx;

               for( idx = 0; idx < su->grid_size && n_with_pair <= 2; idx++)
                  if( su->possibles[group_ptr[idx]] & mask)
                     n_with_pair++;
               if( n_with_pair == 2)
                  {
                  for( idx = 0; idx < su->grid_size; idx++)
                     if( su->possibles[group_ptr[idx]] & mask)
                        if( su->possibles[group_ptr[idx]] & ~mask)
                           {
                           su->possibles[group_ptr[idx]] &= mask;
                           rval = i * su->grid_size + j;
                           }
                  }
               if( rval > 0)
                  return( rval);
               }
   return( -1);
}

void count_bits_in_group( const SUDOKU *su, const int *group_ptr, unsigned *bit_count)
{
   int i;

   for( i = 0; i < su->grid_size; i++)
      if( su->possibles[group_ptr[i]])
         bit_count[i] = count_bits( su->possibles[group_ptr[i]]);
      else
         bit_count[i] = 0;
}

int look_for_naked_pairs_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, k, rval = 0;
   unsigned bit_count[MAX_GRID_SIZE];

   count_bits_in_group( su, group_ptr, bit_count);
   for( i = 1; i < su->grid_size; i++)
      if( su->possibles[group_ptr[i]] && bit_count[i] <= 2)
         for( j = 0; j < i; j++)
            if( su->possibles[group_ptr[j]] && bit_count[j] <= 2)
               {
               const poss_t mask =
                  (su->possibles[group_ptr[i]] | su->possibles[group_ptr[j]]);

               if( count_bits( mask) == 2)
                  for( k = 0; k < su->grid_size; k++)
                     if( k != i && k != j &&
                                (su->possibles[group_ptr[k]] & mask))
                        {
                        su->possibles[group_ptr[k]] &= ~mask;
                        rval = j + i * su->grid_size;
                        }
               if( rval)
                  return( rval);
               }
   return( 0);
}

int look_for_naked_pairs( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_naked_pairs_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

int look_for_hidden_pairs( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_hidden_pairs_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

int look_for_naked_triples_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, k, l, rval = 0;
   unsigned bit_count[MAX_GRID_SIZE];

   count_bits_in_group( su, group_ptr, bit_count);
   for( i = 2; i < su->grid_size; i++)
      if( su->possibles[group_ptr[i]] && bit_count[i] <= 3)
         for( j = 1; j < i; j++)
            if( su->possibles[group_ptr[j]] && bit_count[j] <= 3)
               for( k = 0; k < j; k++)
                  if( su->possibles[group_ptr[k]] && bit_count[k] <= 3)
                     {
                     const poss_t mask =
                           (su->possibles[group_ptr[i]]
                           | su->possibles[group_ptr[j]]
                           | su->possibles[group_ptr[k]]);

                     if( count_bits( mask) == 3)
                        for( l = 0; l < su->grid_size; l++)
                           if( l != i && l != j && l != k &&
                                (su->possibles[group_ptr[l]] & mask))
                              {
                              su->possibles[group_ptr[l]] &= ~mask;
                              rval = k + (j + i * su->grid_size) * su->grid_size;
                              }
                     if( rval)
                     return( rval);
                     }
   return( 0);
}

int look_for_naked_triples( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_naked_triples_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

int look_for_naked_quads_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, k, l, idx, rval = 0;
   unsigned bit_count[MAX_GRID_SIZE];

   count_bits_in_group( su, group_ptr, bit_count);
   for( i = 3; i < su->grid_size; i++)
      if( su->possibles[group_ptr[i]] && bit_count[i] <= 4)
         for( j = 2; j < i; j++)
            if( su->possibles[group_ptr[j]] && bit_count[j] <= 4)
               for( k = 1; k < j; k++)
                  if( su->possibles[group_ptr[k]] && bit_count[k] <= 4)
                     for( l = 1; l < k; l++)
                        if( su->possibles[group_ptr[l]] && bit_count[l] <= 4)
                     {
                     const poss_t mask =
                           (su->possibles[group_ptr[i]]
                           | su->possibles[group_ptr[j]]
                           | su->possibles[group_ptr[k]]
                           | su->possibles[group_ptr[l]]);

                     if( count_bits( mask) == 4)
                        for( idx = 0; idx < su->grid_size; idx++)
                           if( idx != i && idx != j && idx != k &&
                                 idx != l &&
                                (su->possibles[group_ptr[idx]] & mask))
                              {
                              su->possibles[group_ptr[idx]] &= ~mask;
                              rval = l + (k + (j + i * su->grid_size) * su->grid_size) * su->grid_size;
                              }
                     if( rval)
                     return( rval);
                     }
   return( 0);
}

int look_for_naked_quads( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_naked_quads_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

int look_for_hidden_triples_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, k, rval = -1;
   poss_t total_mask = 0;

   for( i = 0; i < su->grid_size; i++)
      total_mask |= su->possibles[group_ptr[i]];
   for( i = 3; i <= su->grid_size; i++)
      if( (total_mask >> i) & 1)
         for( j = 2; j < i; j++)
            if( (total_mask >> j) & 1)
               for( k = 1; k < j; k++)
                  if( (total_mask >> k) & 1)
                     {
                     const poss_t mask
                                  = ((poss_t)1 << i) | ((poss_t)1 << j)
                                  | ((poss_t)1 << k);
                     int n_with_triple = 0, idx;

                     for( idx = 0; idx < su->grid_size && n_with_triple <= 3; idx++)
                        if( su->possibles[group_ptr[idx]] & mask)
                           n_with_triple++;
                     if( n_with_triple == 3)
                        {
                        for( idx = 0; idx < su->grid_size; idx++)
                           if( su->possibles[group_ptr[idx]] & mask)
                              if( su->possibles[group_ptr[idx]] & ~mask)
                                 {
                                 su->possibles[group_ptr[idx]] &= mask;
                                 rval = k + su->grid_size * (j + i * su->grid_size);
                                 }
                        }
                     if( rval > 0)
                        return( rval);
                     }
   return( -1);
}

int look_for_hidden_triples( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_hidden_triples_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

int look_for_hidden_quads_in_group( SUDOKU *su, const int *group_ptr)
{
   int i, j, k, l, rval = -1;
   poss_t total_mask = 0;

   for( i = 0; i < su->grid_size; i++)
      total_mask |= su->possibles[group_ptr[i]];
   for( i = 4; i <= su->grid_size; i++)
      if( (total_mask >> i) & 1)
         for( j = 3; j < i; j++)
            if( (total_mask >> j) & 1)
               for( k = 2; k < j; k++)
                  if( (total_mask >> k) & 1)
                     for( l = 1; l < k; l++)
                        if( (total_mask >> l) & 1)
                     {
                     const poss_t mask = ((poss_t)1 << i) | ((poss_t)1 << j)
                                       | ((poss_t)1 << k) | ((poss_t)1 << l);
                     int n_with_quad = 0, idx;

                     for( idx = 0; idx < su->grid_size && n_with_quad <= 4; idx++)
                        if( su->possibles[group_ptr[idx]] & mask)
                           n_with_quad++;
                     if( n_with_quad == 4)
                        {
                        for( idx = 0; idx < su->grid_size; idx++)
                           if( su->possibles[group_ptr[idx]] & mask)
                              if( su->possibles[group_ptr[idx]] & ~mask)
                                 {
                                 su->possibles[group_ptr[idx]] &= mask;
                                 rval = k + su->grid_size * (j + i * su->grid_size);
                                 }
                        }
                     if( rval > 0)
                        return( rval);
                     }
   return( -1);
}

int look_for_hidden_quads( SUDOKU *su, int *group_idx)
{
   int i;

   for( i = 0; i < n_groups; i++)
      {
      int soln = look_for_hidden_quads_in_group( su, groups + i * su->grid_size);

      if( soln > 0)
         {
         *group_idx = i;
         return( soln);
         }
      }
   return( -1);
}

#ifdef FLIPPING_OK
static void flip_sudoku_diagonally( SUDOKU *su)
{
   int i, j;

   for( i = 0; i < su->n_cells; i++)
      {
      j = (i / su->grid_size) + (i % su->grid_size) * su->grid_size;
      if( j < i)
         {
         const poss_t tpossible = su->possibles[i];
         char tval = su->solutions[i];

         su->solutions[i] = su->solutions[j];
         su->solutions[j] = (char)tval;
         su->possibles[i] = su->possibles[j];
         su->possibles[j] = tpossible;
         tval = block_membership[i];
         block_membership[i] = block_membership[j];
         block_membership[j] = (char)tval;
         }
      }
}
#endif

int look_for_double_pairs( SUDOKU *su, int *group_idx)
{
   int group_number, search_num, rval = -1;

               /* search rows and columns: */
   for( group_number = su->grid_size; group_number < n_groups
                     && rval == -1; group_number++)
      {
      int *tptr = groups + group_number * su->grid_size;

      for( search_num = 1; search_num <= su->grid_size && rval == -1; search_num++)
         {
         int block = -1, i;
         poss_t mask = ((poss_t)1 << search_num);

         for( i = 0; i < su->grid_size && block != -2; i++)
            {
            const int loc = tptr[i];

            if( su->possibles[loc] & mask)
          {
               if( block == -1)
                  block = block_membership[loc];
               else if( block != block_membership[loc])
                  block = -2;    /* two or more blocks share this possible */
               }
            }
         if( block > -1)     /* in this row or column, the only  */
            {           /* possibilities for search_num are in this block */
            int *block_ptr = groups + block * su->grid_size;
            int j;




            for( i = 0; i < su->grid_size; i++)
               if( su->possibles[block_ptr[i]] & mask)
                  {
                  for( j = 0; j < su->grid_size && tptr[j] != block_ptr[i]; j++)
                     ;
                  if( j == su->grid_size)
                     {
                     *group_idx = block;
                     su->possibles[block_ptr[i]] ^= mask;
                     rval = search_num;
                     }
                  }
            }
         }
      }
   return( rval);
}

int look_for_xwings_in_rows( SUDOKU *su)
{
   int i, j, idx, rval = 0;
   int locs[MAX_GRID_SIZE];

   for( idx = 1; idx <= su->grid_size; idx++)
      {
      const poss_t mask = ((poss_t)1 << idx);
      for( i = 0; i < su->grid_size; i++)
         {
         int n_found = 0;

         locs[i] = 0;
         for( j = 0; j < su->grid_size; j++)
            if( su->possibles[i * su->grid_size + j] & mask)
               {
               n_found++;
               locs[i] = locs[i] * su->grid_size + j;
               }
         if( n_found != 2)
            locs[i] = 0;
         }
      for( i = 1; i < su->grid_size; i++)
         for( j = 0; j < i; j++)
            if( locs[i] == locs[j] && locs[i])       /* possible x-wing! */
               {
               int k, x1 = locs[i] % su->grid_size, x2 = locs[i] / su->grid_size;

               for( k = 0; k < su->grid_size; k++)
                  if( k != i && k != j)
                     {
                     if( su->possibles[k * su->grid_size + x1] & mask)
                        {
                        su->possibles[k * su->grid_size + x1] ^= mask;
                        rval = idx;;
                        }
                     if( su->possibles[k * su->grid_size + x2] & mask)
                        {
                        su->possibles[k * su->grid_size + x2] ^= mask;
                        rval = idx;
                        }
                     }
               if( rval)
                  return( rval);
               }
      }
   return( 0);
}

int look_for_xwings_in_columns( SUDOKU *su)
{
   int i, j, idx, rval = 0;
   int locs[MAX_GRID_SIZE];

   for( idx = 1; idx <= su->grid_size; idx++)
      {
      const poss_t mask = ((poss_t)1 << idx);
      for( i = 0; i < su->grid_size; i++)
         {
         int n_found = 0;

         locs[i] = 0;
         for( j = 0; j < su->grid_size; j++)
            if( su->possibles[j * su->grid_size + i] & mask)
               {
               n_found++;
               locs[i] = locs[i] * su->grid_size + j;
               }
         if( n_found != 2)
            locs[i] = 0;
         }
      for( i = 1; i < su->grid_size; i++)
         for( j = 0; j < i; j++)
            if( locs[i] == locs[j] && locs[i])       /* possible x-wing! */
               {
               int k, y1 = locs[i] % su->grid_size, y2 = locs[i] / su->grid_size;

               for( k = 0; k < su->grid_size; k++)
                  if( k != i && k != j)
                     {
                     if( su->possibles[k + su->grid_size * y1] & mask)
                        {
                        su->possibles[k + su->grid_size * y1] ^= mask;
                        rval = idx;;
                        }
                     if( su->possibles[k + su->grid_size * y2] & mask)
                        {
                        su->possibles[k + su->grid_size * y2] ^= mask;
                        rval = idx;
                        }
                     }
               if( rval)
                  return( rval);
               }
      }
   return( 0);
}

/* look_for_double_doubles() looks for situations where,  in two rows (or
columns),  a given number is possible in only two groups in each row.  If
that happens,  the number _cannot_ occur elsewhere within that group.  The
test "if( is_in_groups && !is_in_rows)" means "is this cell in one of those
two groups?  And is it _not_ in one of those two rows?"  If so,  the
search number can be excluded from that cell.

   Similarly,  if there are two groups where the only possibilities for
a number occur in the same two rows (or columns),  then the number must
be located in those groups in those two rows (or columns),  and we can
exclude them from elsewhere in those two rows.  Hence the function
'look_for_double_doubles_part_ii':  same sort of logic,  slightly revised.
*/

int look_for_double_doubles( SUDOKU *su)
{
   int search_num, rval = -1;

// if( verbose > 1)
//    printf( "Looking for double doubles\n");
   for( search_num = 1; rval == -1 && search_num <= su->grid_size; search_num++)
      {
      const poss_t search_mask = ((poss_t)1 << search_num);
      int i, j, k;
      poss_t masks[MAX_GRID_SIZE];

               /* first,  look in rows: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[i / su->grid_size] |= ((poss_t)1 << block_membership[i]);
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 2)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( k = 0; k < su->n_cells; k++)
                     if( su->possibles[k] & search_mask)
                        {
                        const int is_in_rows =
                                (k / su->grid_size == i || k / su->grid_size == j);
                        const poss_t is_in_groups =
                                ((masks[i] >> block_membership[k]) & 1);

                        if( is_in_groups && !is_in_rows)
                           {
                           su->possibles[k] ^= search_mask;
                           rval = search_num;
//                         if( verbose)
//                            printf( "Double double found in rows\n");
                           }
                        }

               /* next,  look in columns: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[i % su->grid_size] |= ((poss_t)1 << block_membership[i]);
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 2)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( k = 0; k < su->n_cells; k++)
                     if( su->possibles[k] & search_mask)
                        {
                        const int is_in_cols =
                                (k % su->grid_size == i || k % su->grid_size == j);
                        const poss_t is_in_groups =
                                ((masks[i] >> block_membership[k]) & 1);

                        if( is_in_groups && !is_in_cols)
                           {
                           su->possibles[k] ^= search_mask;
                           rval = search_num;
//                         if( verbose)
//                            printf( "Double double found in columns\n");
                           }
                        }
      }
   return( rval);
}

int look_for_double_doubles_part_ii( SUDOKU *su)
{
   int search_num, rval = -1;

// if( verbose > 1)
//    printf( "Looking for double doubles II\n");
   for( search_num = 1; rval == -1 && search_num <= su->grid_size; search_num++)
      {
      const poss_t search_mask = ((poss_t)1 << search_num);
      int i, j, k;
      poss_t masks[MAX_GRID_SIZE];

               /* first,  look in rows: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[(int)block_membership[i]] |= ((poss_t)1 << (i / su->grid_size));
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 2)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( k = 0; k < su->n_cells; k++)
                     if( su->possibles[k] & search_mask)
                        {
                        const poss_t is_in_rows =
                              ((masks[i] >> (k / su->grid_size)) & 1);
                        const int is_in_groups = (block_membership[k] == i ||
                                                  block_membership[k] == j);

                        if( !is_in_groups && is_in_rows)
                           {
                           su->possibles[k] ^= search_mask;
                           rval = search_num;
//                         if( verbose)
//                            printf( "Double double found in group/rows\n");
                           }
                        }

               /* next,  look in columns: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[(int)block_membership[i]] |= ((poss_t)1 << (i % su->grid_size));
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 2)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( k = 0; k < su->n_cells; k++)
                     if( su->possibles[k] & search_mask)
                        {
                        const poss_t is_in_cols =
                              ((masks[i] >> (k % su->grid_size)) & 1);
                        const int is_in_groups = (block_membership[k] == i ||
                                                  block_membership[k] == j);

                        if( !is_in_groups && is_in_cols)
                           {
                           su->possibles[k] ^= search_mask;
                           rval = search_num;
//                         if( verbose)
//                            printf( "Double double found in group/cols\n");
                           }
                        }
      }
   return( rval);
}

int look_for_triple_triples( SUDOKU *su)
{
   int search_num, rval = -1;

// if( verbose > 1)
//    printf( "Looking for triple triples\n");
   for( search_num = 1; rval == -1 && search_num <= su->grid_size; search_num++)
      {
      const poss_t search_mask = ((poss_t)1 << search_num);
      int h, i, j, k;
      poss_t masks[MAX_GRID_SIZE];

               /* first,  look in rows: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[i / su->grid_size] |= ((poss_t)1 << block_membership[i]);
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 3)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( h = j + 1; h < su->grid_size; h++)
                     if( masks[i] == masks[h])
                        for( k = 0; k < su->n_cells; k++)
                           if( su->possibles[k] & search_mask)
                              {
                              const int is_in_rows =
                                      (k / su->grid_size == i || k / su->grid_size == j
                                    || k / su->grid_size == h);
                              const poss_t is_in_groups =
                                      ((masks[i] >> block_membership[k]) & 1);

                              if( is_in_groups && !is_in_rows)
                                 {
                                 su->possibles[k] ^= search_mask;
                                 rval = search_num;
//                               if( verbose)
//                                  printf( "Triple triple found in rows\n");
                                 }
                              }

               /* next,  look in columns: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[i % su->grid_size] |= ((poss_t)1 << block_membership[i]);
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 3)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( h = j + 1; h < su->grid_size; h++)
                     if( masks[i] == masks[h])
                        for( k = 0; k < su->n_cells; k++)
                           if( su->possibles[k] & search_mask)
                              {
                              const int is_in_cols =
                                      (k % su->grid_size == i || k % su->grid_size == j
                                    || k % su->grid_size == h);
                              const poss_t is_in_groups =
                                      ((masks[i] >> block_membership[k]) & 1);

                              if( is_in_groups && !is_in_cols)
                                 {
                                 su->possibles[k] ^= search_mask;
                                 rval = search_num;
//                               if( verbose)
//                                  printf( "Triple triple found in columns\n");
                                 }
                              }
      }
   return( rval);
}

int look_for_triple_triples_part_ii( SUDOKU *su)
{
   int search_num, rval = -1;

// if( verbose > 1)
//    printf( "Looking for triple triples II\n");
   for( search_num = 1; rval == -1 && search_num <= su->grid_size; search_num++)
      {
      const poss_t search_mask = ((poss_t)1 << search_num);
      int h, i, j, k;
      poss_t masks[MAX_GRID_SIZE];

               /* first,  look in rows: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[(int)block_membership[i]] |= ((poss_t)1 << (i / su->grid_size));
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 3)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( h = j + 1; h < su->grid_size; h++)
                     if( masks[i] == masks[h])
                        for( k = 0; k < su->n_cells; k++)
                           if( su->possibles[k] & search_mask)
                              {
                              const poss_t is_in_rows =
                                    ((masks[i] >> (k / su->grid_size)) & 1);
                              const int is_in_groups = (block_membership[k] == i ||
                                                        block_membership[k] == h ||
                                                        block_membership[k] == j);

                              if( !is_in_groups && is_in_rows)
                                 {
                                 su->possibles[k] ^= search_mask;
                                 rval = search_num;
//                               if( verbose)
//                                  printf( "Triple triple found in group/rows\n");
                                 }
                              }

               /* next,  look in columns: */
      for( i = 0; i < su->grid_size; i++)
         masks[i] = 0;
      for( i = 0; i < su->n_cells; i++)
         if( su->possibles[i] & search_mask)
            masks[(int)block_membership[i]] |= ((poss_t)1 << (i % su->grid_size));
      for( i = 0; i < su->grid_size; i++)
         if( count_bits( masks[i]) == 3)
            for( j = i + 1; j < su->grid_size; j++)
               if( masks[i] == masks[j])
                  for( h = j + 1; h < su->grid_size; h++)
                     if( masks[i] == masks[h])
                        for( k = 0; k < su->n_cells; k++)
                           if( su->possibles[k] & search_mask)
                              {
                              const poss_t is_in_cols =
                                    ((masks[i] >> (k % su->grid_size)) & 1);
                              const int is_in_groups = (block_membership[k] == i ||
                                                        block_membership[k] == j ||
                                                        block_membership[k] == h);


                              if( !is_in_groups && is_in_cols)
                                 {
                                 su->possibles[k] ^= search_mask;
                                 rval = search_num;
//                               if( verbose)
//                                  printf( "Triple triple found in group/cols\n");
                                 }
                              }
      }
   return( rval);
}


int look_for_xwings( SUDOKU *su)
{
   int rval = look_for_xwings_in_rows( su);

   if( !rval)
      rval = look_for_xwings_in_columns( su);

   return( rval);
}

static int load_sudoku_grid( SUDOKU *su, const char *filename)
{
   int i, j, rval = 0;
   FILE *ifile = fopen( filename, "rb");
   char buff[80];

   if( ifile)
      {
      for( i = 0; i < su->n_cells; i++)
         {
         su->possibles[i] = all_possibles;
         su->solutions[i] = 0;
         }
      for( i = 0; i < su->grid_size; i++)
         {
         if( !fgets( buff, sizeof( buff), ifile))
            {
            fclose( ifile);
            return( -2);
            }
         if( *buff == '-')       /* separator line */
            if( !fgets( buff, sizeof( buff), ifile))
               {
               fclose( ifile);
               return( -3);
               }
         for( j = strlen( buff); j < 60; j++)      /* pad with spaces */
            buff[j] = ' ';
         for( j = 0; j < su->grid_size; j++)
            if( buff[j * 2] >= '1' && buff[j * 2] <= '9')
               {
               su->solutions[j + i * su->grid_size] = (char)( buff[j * 2] - '0');
               set_possibles( su, j + i * su->grid_size);
               }
         }
      fclose( ifile);
      }
   else
      rval = -1;
   return( rval);
}


static void show_possibility_masks( const poss_t *possibles, const int grid_size)
{
   int j;
   static const char *formats[4] = { " %01x", " %02x", " %03x", " %04x" };

   printf( "  ");
   for( j = 0; j < grid_size; j++)
      printf( formats[(grid_size - 1)/ 4], (unsigned)possibles[j] >> 1);
}


      /* Enough characters to represent up to a 60x60 Sudoku:  ten digits, */
      /* 24 uppercase letters (I and O omitted to evade confusion with 1   */
      /* and 0),  and 26 lowercase letters.  The program can only make     */
      /* Sudoku up to 60x60,  unless revised to make use of 64-bit ints.   */
static const char *character_representation =
          " 1234567890ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

void show_sudoku_grid( SUDOKU *su)
{
   int i, j, loc = 0, is_completed_grid = 1;

   for( i = 0; i < su->n_cells; i++)
      if( !su->solutions[i])
         is_completed_grid = 0;

   printf( " ");
   for( i = 0; i < su->grid_size; i++)
      printf( " %c", (char)( 'a' + i));
   printf( "\n");
   for( i = 0; i < su->grid_size; i++)
      {
      int show_separator = 1;

      printf( "%c ", (i < 9) ? '1' + i : 'a' + i - 9);
      for( j = 0; j < su->grid_size; j++, loc++)
         {
         if( j)
            {
            if( block_membership[loc - 1] != block_membership[loc])
               printf( "|");
            else
               printf( " ");
            }
         if( i == su->grid_size - 1 || block_membership[loc] ==
                                   block_membership[loc + su->grid_size])
            show_separator = 0;
         printf( "%c", character_representation[(int)su->solutions[loc]]);
         }

      if( !is_completed_grid && su->grid_size < 13)
            show_possibility_masks( su->possibles + i * su->grid_size,
                                                        su->grid_size);
      printf( "\n");
      if( show_separator)
         {
         printf( "  ");
         for( j = 0; j < su->grid_size; j++)
            if( j == su->grid_size - 1)
               printf( "-\n");
            else if( block_membership[j + i * su->grid_size] !=
                     block_membership[j + i * su->grid_size + 1])
               printf( "-+");
            else
               printf( "--");
         }
      }
}

static int max_acceptable_method = 0;

int make_simplified_guess( SUDOKU *su)
{
   int i, j, rval = -1;

   for( i = 0; i < su->n_cells && rval == -1; i++)
      if( count_bits( su->possibles[i]) == 2)
         {
         int count = 0, soln0, soln1;
         SUDOKU *guesses[2];
         int old_max_acceptable_method = max_acceptable_method;

         guesses[0] = generate_blank_sudoku( su->grid_size);
         guesses[1] = generate_blank_sudoku( su->grid_size);
         max_acceptable_method = 4;       /* stick to simpler methods */
         for( j = 1; j <= su->grid_size; j++)
            if( (su->possibles[i] >> j) & 1)
               {
               copy_sudoku( guesses[count], su);
               guesses[count]->solutions[i] = (char)j;
               set_possibles( guesses[count], i);
               count++;
               if( count > 2)
                  exit( -1);
               }
         if( count != 2)
            exit( -1);
         soln0 = solve_sudoku( guesses[0]);
         soln1 = solve_sudoku( guesses[1]);
         max_acceptable_method = old_max_acceptable_method;
                  /* OK,  now we've tried both possibilities.  Did we find */
                  /* that no matter which is used,  something turns out the */
                  /* same either way? */
         for( j = 0; j < su->n_cells; j++)
            if( !su->solutions[j])
               {
               poss_t mask;

               if( guesses[0]->solutions[j] == guesses[1]->solutions[j] &&
                         guesses[0]->solutions[j])
                  {
                  rval = i;
                  su->solutions[j] = guesses[0]->solutions[j];
                  set_possibles( su, j);
                  }
               else if( guesses[0]->possibles[j] != su->possibles[j])
                  {
                  poss_t mask0, mask1;

                  if( guesses[0]->solutions[j])
                     mask0 = ((poss_t)1 << guesses[0]->solutions[j]);
                  else
                     mask0 = guesses[0]->possibles[j];
                  if( guesses[1]->solutions[j])
                     mask1 = ((poss_t)1 << guesses[1]->solutions[j]);
                  else
                     mask1 = guesses[1]->possibles[j];
                  for( mask = 1; mask; mask <<= 1)
                     if( su->possibles[j] & mask)
                        if( !(mask0 & mask) && !(mask1 & mask))
                           {
                           rval = i;
                           su->possibles[j] ^= mask;
                           }
                  }
               }
         free( guesses[0]);
         free( guesses[1]);
         }
   return( rval);
}

/* make_a_guess( ) returns MULTIPLE_SOLUTIONS_FOUND if multiple solutions
   are found,  -1 if no solution is found. */

static int multiple_solutions = 0;
static SUDOKU *an_example_solution;

int make_a_guess( SUDOKU *su)
{
   int j, i, best = 1000, best_idx = 0, n_cells_found = 0, the_answer = 0;
   const int random_start = rand( ) % su->grid_size;
   SUDOKU *su_new;

   if( multiple_solutions)
      {
      copy_sudoku( su, an_example_solution);
      return( MULTIPLE_SOLUTIONS_FOUND);
      }

            /* First: which cell has the least number of possibilities */
            /* to consider?  That cell will be the best one to guess at. */
            /* Of course,  if there are only two possibilities,  we might */
            /* as well stop where we are. */
   for( i = 0; i < su->n_cells; i++)
      if( su->possibles[i] && best != 2)
         {
         int n_possibles = 0;

         n_cells_found++;
         for( j = 1; j <= su->grid_size; j++)
            if( (su->possibles[i] >> j) & 1)
               n_possibles++;
         if( n_possibles < best)
            {
            best = n_possibles;
            best_idx = i;
            }
         }
#ifdef _MSC_VER
   if( kbhit( ))
      verbose = getch( ) - '0';
#endif
   if( verbose)
      {
      poss_t crc = 0, crc2 = 0;

      for( i = 0; i < su->n_cells; i++)
         {
         crc ^= (su->possibles[i] << (i % 4));
         crc2 ^= (su->solutions[i] << (i % 7));
         }
      printf( "Making complete guess: cell %d, %d possibles; %lx %lx\n",
                     best_idx, best, (unsigned long)crc, (unsigned long)crc2);
      }
   if( verbose > 1)
      show_sudoku_grid( su);
   su_new = generate_blank_sudoku( su->grid_size);
   for( j = 0; !multiple_solutions && j < su->grid_size; j++)
      {
      const int trial_answer = (random_start + j) % su->grid_size + 1;

      if( (su->possibles[best_idx] >> trial_answer) & 1)
         {
         copy_sudoku( su_new, su);
         if( su_new->solutions[best_idx])
            printf( "??? best_idx %d, soln is %d\n", best_idx,
                           su_new->solutions[best_idx]);
         if( trial_answer < 1 || trial_answer > su->grid_size)
            printf( "??? trial_answer = %d\n", trial_answer);
         if( verbose)
            printf( "Trying %d in the cell\n", trial_answer);
         su_new->solutions[best_idx] = (char)trial_answer;
         set_possibles( su_new, best_idx);
         if( solve_sudoku( su_new) == 1)
            {
            if( the_answer)
               {
               multiple_solutions = 1;
               if( !an_example_solution)
                  an_example_solution = generate_blank_sudoku( su->grid_size);
               copy_sudoku( su, su_new);
               copy_sudoku( an_example_solution, su_new);
               free( su_new);
               printf( "Got multiples\n");
               return( MULTIPLE_SOLUTIONS_FOUND);
               }
            else
               {
               printf( "Got the answer\n");
               the_answer = trial_answer;
               }
            }
         }
      }
   free( su_new);
   if( the_answer)
      {
      su->solutions[best_idx] = (char)the_answer;
      set_possibles(su, best_idx);
      return( best_idx);
      }
   return( PUZZLE_IS_UNSOLVABLE);
}

/* take_sudoku_step( ) returns PUZZLE_IS_UNSOLVABLE if no solution
is possible (either because an internal inconsistency was found --
"there's no valid value for this cell",  for example" -- or because
we didn't look hard enough;  the 'max_acceptable_method' was set too
low to find the solution).  Or it returns MULTIPLE_SOLUTIONS_FOUND
if there are multiple solutions,  or a value >= 0 if a step has been
taken (meaning of the rval then depends on the 'method' used). */

int excluded_methods = 0;

int take_sudoku_step( SUDOKU *su, int *group_idx, int *method)
{
   int soln;

   if( multiple_solutions)
      {
      *method = -3;
      return( MULTIPLE_SOLUTIONS_FOUND);
      }

   soln = look_for_singletons( su, group_idx);
   if( soln >= 0)
      {
      *method = 1;
      return( soln);
      }

   if( soln == PUZZLE_IS_UNSOLVABLE || max_acceptable_method == 1)      /* insolvable case */
      {
      *method = -2;
      return( PUZZLE_IS_UNSOLVABLE);
      }

   soln = look_for_single_candidates( su);
   if( soln >= 0)
      {
      *method = 2;
      return( soln);
      }
   if( max_acceptable_method == 2)
      return( PUZZLE_IS_UNSOLVABLE);

   soln = look_for_candidate_lines( su, group_idx);
   if( soln >= 0)
      {
      *method = 3;
      return( soln);
      }
   if( max_acceptable_method == 3)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x10))
      soln = look_for_double_pairs( su, group_idx);
   if( soln > 0)
      {
      *method = 4;
      return( soln);
      }
   if( max_acceptable_method == 4)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x20))
      soln = look_for_naked_pairs( su, group_idx);
   if( soln >= 0)
      {
      *method = 5;
      return( soln);
      }
   if( max_acceptable_method == 5)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x40))
      soln = look_for_naked_triples( su, group_idx);
   if( soln >= 0)
      {
      *method = 6;
      return( soln);
      }
   if( max_acceptable_method == 6)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x80))
      soln = look_for_naked_quads( su, group_idx);
   if( soln >= 0)
      {
      *method = 7;
      return( soln);
      }
   if( max_acceptable_method == 7)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x100))
      soln = look_for_hidden_pairs( su, group_idx);
   if( soln >= 0)
      {
      *method = 8;
      return( soln);
      }
   if( max_acceptable_method == 8)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x200))
      soln = look_for_hidden_triples( su, group_idx);
   if( soln >= 0)
      {
      *method = 9;
      return( soln);
      }
   if( max_acceptable_method == 9)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x400))
      soln = look_for_hidden_quads( su, group_idx);
   if( soln >= 0)
      {
      *method = 10;
      return( soln);
      }
   if( max_acceptable_method == 10)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x800))
      {
      soln = look_for_xwings( su);
      if( soln)
         {
         *method = 11;
         return( soln);
         }
      }

   if( max_acceptable_method == 11)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x8000))
      soln = make_simplified_guess( su);
   if( soln >= 0)
      {
      *method = 12;
      return( soln);
      }

   if( max_acceptable_method == 12)
      return( PUZZLE_IS_UNSOLVABLE);

   if( !(excluded_methods & 0x2000))
      if( su->grid_size > 9)   /* won't find double doubles in smaller grids */
         {
         soln = look_for_double_doubles( su);
         if( soln >= 0)
            {
            *method = 13;
            return( soln);
            }

         soln = look_for_double_doubles_part_ii( su);
         if( soln >= 0)
            {
            *method = 14;
            return( soln);
            }
         }

   if( !(excluded_methods & 0x4000))
      if( su->grid_size > 14)      /* won't find triple triples in smaller grids */
         {
         soln = look_for_triple_triples( su);
         if( soln >= 0)
            {
            *method = 15;
            return( soln);
            }
         soln = look_for_triple_triples_part_ii( su);
         if( soln >= 0)
            {
            *method = 16;
            return( soln);
            }
         }

// printf( "Making a guess...");
   soln = make_a_guess( su);
// printf( "soln %d\n", soln);
   if( soln >= 0)
      {
      *method = 99;
      return( soln);
      }
   if( soln == PUZZLE_IS_UNSOLVABLE)           /* no guess led to a solution */
      return( PUZZLE_IS_UNSOLVABLE);
   if( soln == MULTIPLE_SOLUTIONS_FOUND)         /* multiple solutions */
      {
      *method = -3;
//    printf( "Multiple solutions! (2)\n");
//    show_sudoku_grid( su);
      return( MULTIPLE_SOLUTIONS_FOUND);
      }

   if( soln == -1)         /* no answer possible */
      {
      *method = -3;
//    printf( "No solution! (5)\n");
//    show_sudoku_grid( su);
      return( -1);
      }

               /* In theory,  we should never get here:   */
   printf( "Soln = %d;  dunno what it means\n", soln);
   *method = -1;
   return( -1);
}

static int max_method_used;

/* solve_sudoku( ) returns 1 if the grid has been completed,
PUZZLE_IS_UNSOLVABLE if the puzzle is not solvable,
MULTIPLE_SOLUTIONS_FOUND if it has multiple solutions. */

int solve_sudoku( SUDOKU *su)
{
   int i, n_cells_solved, soln, group_idx, method, rval = -1;
   int n_iterations = 0;
   const int switchover_freq = su->n_cells * 3;

   if( multiple_solutions)
      return( MULTIPLE_SOLUTIONS_FOUND);
   while( rval < 0 && rval != MULTIPLE_SOLUTIONS_FOUND
                          && rval != PUZZLE_IS_UNSOLVABLE)
      {
               /* count the cells;  maybe we're done? */
      for( i = n_cells_solved = 0; i < su->n_cells; i++)
         if( su->solutions[i])
            n_cells_solved++;
      if( verbose > 7)
         {
         printf( "rval %d, n_iterations %d; %d\n", rval, n_iterations, n_cells_solved);
         show_sudoku_grid( su);
         }
      if( n_cells_solved == su->n_cells)
         rval = 1;
      else        /* nope,  still got work to do */
         {
         const int saved_excluded_methods = excluded_methods;

               /* If we've been going on for an extra-special long time, */
               /* we should probably try the full arsenal of solving tools. */
               /* Some of those tools are usually slow,  but if there's no */
               /* other way to do it,  we use 'em.                         */
         if( n_iterations % switchover_freq > switchover_freq * 3 / 4)
            {
//          printf( "%d; ", n_cells_solved);
            excluded_methods = 0;
            }
         n_iterations++;
         soln = take_sudoku_step( su, &group_idx, &method);
         excluded_methods = saved_excluded_methods;
         if( max_acceptable_method == 0 && verbose > 2)
            {
            printf( "%d cells; soln %d, method %d\n", n_cells_solved, soln, method);
            if( multiple_solutions)
               printf( "Multiples already found\n");
            }
         if( method > max_method_used)
            max_method_used = method;
//       if( method == -1)       // What's this doing here?  Method should
//           rval = 0;           // _never_ be -1... ?! */
         if( soln == PUZZLE_IS_UNSOLVABLE)      /* unsolvable */
            {
//          printf( "Insoluble (7)\n");
//          show_sudoku_grid( su);
            rval = PUZZLE_IS_UNSOLVABLE;
            }
         if( soln == MULTIPLE_SOLUTIONS_FOUND)
            {
//          printf( "Yes,  it's really multiple\n");
//          show_sudoku_grid( su);
            rval = MULTIPLE_SOLUTIONS_FOUND;
            }
         }
      }
   return( rval);
}

static void log_sudoku_grid( FILE *log_file, const SUDOKU *su)
{
   int i, j, loc = 0, n_cells_filled = 0;
   char format_text[20];

   for( i = 0; i < su->n_cells; i++)
      if( su->solutions[i])
         n_cells_filled++;
   fprintf( log_file, "%2d %3d\n", su->grid_size, n_cells_filled);
   strcpy( format_text, " %0!x");
   format_text[3] = (char)('0' + (su->grid_size + 3) / 4);
   for( i = 0; i < su->grid_size; i++, loc += su->grid_size)
      {
      for( j = 0; j < su->grid_size; j++)
         fprintf( log_file, "%c ",
                 character_representation[(int)su->solutions[loc + j]]);
      if( n_cells_filled != su->n_cells)
         {
         fprintf( log_file, "   ");
         for( j = 0; j < su->grid_size; j++)
            fprintf( log_file, format_text, (unsigned)su->possibles[loc + j] >> 1);
         }
      fprintf( log_file, "\n");
      }
}

static void create_randomly_filled_grid( SUDOKU *su)
{
   int i;

   memset( su->solutions, 0, su->n_cells);
   memset( su->possibles, 0, su->n_cells * sizeof( poss_t));

            /* In theory,  we could just "solve" a blank Sudoku.  But */
            /* it's just a hair faster if the first su->grid_size cells are  */
            /* set,  and will be just as random.  So: */
   for( i = 0; i < su->grid_size; i++)
      su->solutions[i] = (char)( i + 1);
   if( verbose > 8)
      printf( "Top line set, ");
   for( i = su->grid_size - 1; i; i--)
      {
      const int idx = rand( ) % i;

      if( verbose > 8)
         printf( "Swapping %d with %d\n", idx, i);
      SWAP_INTEGER_VALUES( su->solutions[idx], su->solutions[i]);
      }
   if( verbose > 8)
      printf( "randomized\n");
   check_sudoku( su);
   set_all_possibles( su);
   check_sudoku( su);
   if( verbose > 8)
      printf( "Possibles set\n");
   max_acceptable_method = 0;    /* "anything goes" */
   solve_sudoku( su);
   check_sudoku( su);
   check_sudoku( an_example_solution);
   copy_sudoku( su, an_example_solution);
}

static int generate_sudoku( SUDOKU *result, SUDOKU *su, const int acceptable_methods)
{
   int i, j, max_method_in_soln = 0, n_iterations = 0;
   int search_len = (su->n_cells + 1) / 2;
   int *search_order = (int *)calloc( search_len, sizeof( int));
   SUDOKU *temp_su, *temp_result;

   if( verbose > 1)
      printf( "In generate_sudoku()\n");
   check_sudoku( result);
   check_sudoku( su);
   create_randomly_filled_grid( result);
   check_sudoku( result);
   if( verbose > 1)
      show_sudoku_grid( result);
   for( i = 0; i < search_len; i++)
      search_order[i] = i;
   max_acceptable_method = acceptable_methods;
   temp_su = generate_blank_sudoku( su->grid_size);
   check_sudoku( temp_su);
   temp_result = generate_blank_sudoku( su->grid_size);
   check_sudoku( temp_result);
   while( max_method_in_soln != max_acceptable_method)
      {
      copy_sudoku( su, result);
      for( i = search_len - 1; i; i--)    /* rearrange the search order */
         {
         j = rand( ) % i;
         SWAP_INTEGER_VALUES( search_order[j], search_order[i]);
         }

      for( i = 0; i < search_len; i++)
         {
         check_sudoku( su);
         copy_sudoku( temp_su, su);
         check_sudoku( temp_su);
         temp_su->solutions[search_order[i]] = 0;
         temp_su->solutions[su->n_cells - 1 - search_order[i]] = 0;
         set_all_possibles( temp_su);
         check_sudoku( temp_su);
         multiple_solutions = 0;
         max_method_used = 0;
         copy_sudoku( temp_result, temp_su);
         check_sudoku( temp_result);
         if( solve_sudoku( temp_result) == 1)
            if( !multiple_solutions)    /* yes,  we can remove these cells */
               if( max_method_used <= acceptable_methods)
                  {
                  max_method_in_soln = max_method_used;
                  copy_sudoku( su, temp_su);
                  }
         }
      n_iterations++;
      }
   free( search_order);
   free( temp_su);
   free( temp_result);
   return( n_iterations);
}

static void put_supergrid( FILE *ofile, const int grid_size)
{
   int x, y;

               /* Show horizontal lines: */
   for( y = 0; y < grid_size - 1; y++)
      {
      const char *tptr = block_membership + y * grid_size;

      for( x = 0; x < grid_size; x++)
         if( tptr[x] != tptr[x + grid_size])
            {
            fprintf( ofile, "%d %d ", x, grid_size - y - 1);
            while( x < grid_size && tptr[x] != tptr[x + grid_size])
               x++;
            fprintf( ofile, "%d %d show_sub_line\n", x, grid_size - y - 1);
            }
      }
               /* Now show vertical lines: */
   for( x = 0; x < grid_size - 1; x++)
      {
      const char *tptr = block_membership + x;

      for( y = 0; y < grid_size; y++, tptr += grid_size)
         if( tptr[0] != tptr[1])
            {
            fprintf( ofile, "%d %d ", x + 1, grid_size - y);
            while( y < grid_size && tptr[0] != tptr[1])
               {
               y++;
               tptr += grid_size;
               }
            fprintf( ofile, "%d %d show_sub_line\n", x + 1, grid_size - y);
            }
      }
}

char *fgets_trimmed( char *str, int n, FILE *fptr)
{
   if( !fgets( str, n, fptr))
      {
      *str = '\0';
      str = NULL;
      }
   else
      {
      int i;

      for( i = 0; str[i] != 10 && str[i] != 13; i++)
         ;
      str[i] = '\0';
      }
   return( str);
}

int load_grid_data( const char *grid_name)
{
   FILE *ifile = fopen( "sud_grp.txt", "rb");
   char buff[200];
   int rval = -1, grid_size, n_cells;

   while( fgets_trimmed( buff, sizeof( buff), ifile))
      if( !memcmp( buff, "GROUP ", 6) && !strcmp( buff + 6, grid_name))
         {
         int i, j, loc = 0, inconsistencies_found = 0;
         int n_across = 0, spin = 0;

         if( !fgets_trimmed( buff, sizeof( buff), ifile))
            {
            fclose( ifile);
            return( -1);
            }
         sscanf( buff, "%d %d %d", &grid_size, &n_across, &spin);
         n_cells = grid_size * grid_size;
         block_membership = (char *)malloc( n_cells);
         if( n_across)        /* this is a "regular" grid */

            {
            const int n_vert = grid_size / n_across;

            for( i = 0; i < grid_size; i++)
               for( j = 0; j < grid_size; j++)
                  block_membership[loc++] = (char)( i / n_vert +
                           n_across * (j / n_across));
            if( spin)
               {
               int i1, j1;

               for( i = 0; i < grid_size; i++)
                  for( j = 0; j < grid_size; j++)
                     if( (i1 = i % spin) > (j1 = j % spin) &&
                           ((i / spin) ^ (j / spin)) & 1)
                        {
                        const int loc1 = i + j * grid_size;
                        const int loc2 = (i + j1 - i1) + (j + i1 - j1) * grid_size;
                        const char tchar = block_membership[loc1];

                        block_membership[loc1] = block_membership[loc2];
                        block_membership[loc2] = tchar;
                        }
               }
            }
         else     /* Irregular grid, specified by grid_size lines in file: */
            for( i = 0; i < grid_size; i++)
               {
               if( !fgets_trimmed( buff, sizeof( buff), ifile))
                  {
                  fclose( ifile);
                  return( -9);
                  }
               for( j = 0; j < grid_size; j++)
                  block_membership[loc++] = (char)atoi( buff + j * 3);
               }
                     /* check block memberships for internal consistency: */
         for( i = 0; i < grid_size; i++)
            {
            int count = 0;

            for( j = 0; j < n_cells; j++)
               if( block_membership[j] == i)
                   count++;
            if( count != grid_size)
               {
               printf( "There are %d members of group %d\n", count, i);
               inconsistencies_found = 1;
               }
            }
         if( inconsistencies_found)
            rval = PUZZLE_IS_UNSOLVABLE;
         else
            {
            n_groups = grid_size * 3;
            all_possibles = ((poss_t)2 << grid_size) - (poss_t)2;
            set_groups( groups, block_membership, grid_size);
            rval = grid_size;
            }
         }
   fclose( ifile);
   return( rval);
}

#define MAX_PUZZLES 100

int main( int argc, char **argv)
{
   SUDOKU *su;
   int n_repeats = 1, is_x_sudoku = 0, grid_size;
   int soln = 0, step = 0, n_cells_solved = 0, desired_method = 1, i;
               /* 64K puzzles should be enough for anybody */
               /* (though any seed up to 4G can be selected with '-r'): */
   unsigned seed = time( NULL) % 0xffff;
   const char *template_filename = "template.ps";
   const char *output_filename = "sudoku.ps";
   const char *input_sudoku_filename = NULL;
   const char *log_filename = "sudoku.log";
   const char *grid_name = "9";

   setvbuf( stdout, NULL, _IONBF, 0);
   for( i = 1; i < argc; i++)
      if( argv[i][0] == '-')
         switch( argv[i][1])
            {
            case 't':
               template_filename = argv[i] + 2;
               break;
            case 'o':
               output_filename = argv[i] + 2;
               break;
            case 'i':
               input_sudoku_filename = argv[i] + 2;
               break;
            case 'l':
               log_filename = argv[i] + 2;
               break;
            case 'v':
               verbose = atoi( argv[i] + 2);
               printf( "Verbosity level set to %d\n", verbose);
               break;
            case 'n':
               n_repeats = atoi( argv[i] + 2);
               break;
            case 'r':
               sscanf( argv[i] + 2, "%x", &seed);
               break;
            case 'm':
               desired_method = atoi( argv[i] + 2);
               printf( "Using solution method %d\n", desired_method);
               break;
            case 'c':
               character_representation = argv[i] + 2;
               break;
            case 'g':
               grid_name = argv[i] + 2;
               printf( "Grid name set to %s\n", grid_name);
               break;
            case 'x':
               is_x_sudoku = 1;
               printf( "Including diagonals in Sudoku rules\n");
               break;
            case 'X':
               sscanf( argv[i] + 2, "%x", &excluded_methods);
               break;
            }

   grid_size = load_grid_data( grid_name);
   if( grid_size <= 0)
      {
      printf( "Grid '%s' not found\n", grid_name);
      exit( -1);
      }
   if( is_x_sudoku)        /* include diagonals for "x-sudoku" */
      n_groups += 2;
   srand( seed);
   if( !input_sudoku_filename)   /* we're generating a/many sudoku, */
      {                          /* not solving one */
      FILE *ifile = fopen( template_filename, "rb");
      FILE *ofile = fopen( output_filename, "wb");
      FILE *log_file = fopen( log_filename, "wb");
      char buff[200];
      SUDOKU **puzzles = (SUDOKU **)calloc( MAX_PUZZLES, sizeof( SUDOKU *));
      SUDOKU **solutions = (SUDOKU **)calloc( MAX_PUZZLES, sizeof( SUDOKU *));
      long tenths_of_seconds, starting_offset = 0;
      int page_number = 0, rep;

      if( !ifile)
         {
         printf( "Couldn't open template file '%s'\n", template_filename);
         exit( -1);
         }
      if( !ofile)
         {
         printf( "Couldn't open output file '%s'\n", output_filename);
         exit( -1);
         }
      if( !solutions || !puzzles)
         {
         printf( "Couldn't allocate puzzle/solution grids\n");
         exit( -1);
         }
      setvbuf( ofile, NULL, _IONBF, 0);
      for( rep = 0; rep < n_repeats; rep++)
         {
         fseek( ifile, starting_offset, 0L);

         while( fgets( buff, sizeof( buff), ifile))
            if( !memcmp( buff, "%%Pages:", 8))
               {
               sprintf( buff + 8, " %d\n", atoi( buff + 8) * n_repeats);
               fputs( buff, ofile);
               }
            else if( !memcmp( buff, "%%Page:", 7))
               {
               if( !starting_offset)
                  starting_offset = ftell( ifile) - strlen( buff);
               page_number++;
               sprintf( buff + 7, " %d %d\n", page_number, page_number);
               fputs( buff, ofile);
               }
            else if( !memcmp( buff, "(pg)", 4))
               {
               char tbuff[80];

               sprintf( tbuff, "(%d)%s", page_number, buff + 4);
               fputs( tbuff, ofile);
               }
            else if( memcmp( buff, "% Sudoku", 8) && memcmp( buff, "% Soluti", 8))
               {
               char *text_replace = strstr( buff, "%RANDOM_SEED%");

               if( text_replace)
                  {
                  char tbuff[100];

                  *text_replace = '\0';
                  sprintf( tbuff, "%s%x%s", buff, seed, text_replace + 13);
                  strcpy( buff, tbuff);
                  }
               text_replace = strstr( buff, "%LEVEL%");
               if( text_replace)
                  {
                  sprintf( text_replace, "%-6d", desired_method);
                  text_replace[6] = ' ';
                  }

               text_replace = strstr( buff, "%FULL_TIME_____________%");
               if( text_replace)
                  {
                  time_t t0 = time( NULL);

                  memcpy( text_replace, ctime( &t0), 24);
                  }

               if( !memcmp( buff, "/grid  %%", 9))
                  {
                  buff[7] = (char)( '0' + grid_size / 10);
                  buff[8] = (char)( '0' + grid_size % 10);
                  }

               if( !memcmp( buff, "%PUT_SUPERGRID%", 15))
                  {
                  *buff = '\0';
                  put_supergrid( ofile, grid_size);
                  }

               fputs( buff, ofile);
               }
            else
               {
               const int puzzle_number = atoi( buff + 8);
               int i;

               printf( "Puzzle %d\n", puzzle_number);
               if( !puzzles[puzzle_number])
                  {           /* haven't created this puzzle yet: */
                  int n_iterations, n_cells_solved = 0;

                  puzzles[puzzle_number] = generate_blank_sudoku( grid_size);
                  solutions[puzzle_number] = generate_blank_sudoku( grid_size);
                  if( verbose > 2)
                     printf( "Generating sudoku...\n");
                  n_iterations = generate_sudoku( solutions[puzzle_number],
                                      puzzles[puzzle_number], desired_method);

                  if( verbose)
                     {
                     show_sudoku_grid( solutions[puzzle_number]);
                     show_sudoku_grid( puzzles[puzzle_number]);
                     }
                  log_sudoku_grid( log_file, puzzles[puzzle_number]);
                  log_sudoku_grid( log_file, solutions[puzzle_number]);
                  for( i = n_cells_solved = 0; i < grid_size * grid_size; i++)
                     if( puzzles[puzzle_number]->solutions[i])
                        n_cells_solved++;
                  printf( "%d cells; %d iterations\n", n_cells_solved, n_iterations);
                  }
               for( i = 0; i < grid_size; i++)
                  {
                  int j;
                  SUDOKU *sud = (buff[3] == 'u' ?
                                     puzzles : solutions)[puzzle_number];

                  fprintf( ofile, "(");
                  for( j = 0; j < grid_size; j++)
                     {
                     int oval = sud->solutions[j + i * grid_size];

                     fprintf( ofile, "%c", character_representation[oval]);
                     }
                  fprintf( ofile, ")\n");
                  }
               }
         for( i = 0; i < MAX_PUZZLES; i++)
            if( puzzles[i])
               {
               free( puzzles[i]);
               free( solutions[i]);
               solutions[i] = puzzles[i] = NULL;
               }
         }
      fclose( ifile);
      fclose( ofile);
      tenths_of_seconds = clock( ) * 10 / CLOCKS_PER_SEC;
      printf( "%ld.%ld seconds elapsed\n", tenths_of_seconds / 10,
                                         tenths_of_seconds % 10);
#if 0
      tenths_of_seconds = test_t * 10 / CLOCKS_PER_SEC;
      printf( "Test: %d.%d seconds elapsed\n", tenths_of_seconds / 10,
                                         tenths_of_seconds % 10);
#endif
      exit( -1);
      }
   su = generate_blank_sudoku( grid_size);
   load_sudoku_grid( su, input_sudoku_filename);
   show_sudoku_grid( su);
   while( soln >= 0 && n_cells_solved != su->n_cells)
      {
      int group_idx, i, method;

      soln = take_sudoku_step( su, &group_idx, &method);
      switch( method)
         {
         case 1:
            printf( "Singleton found at %c%d in %s:\n",
                     "abcdefghijklmnopqrst"[soln % su->grid_size],
                     soln / su->grid_size + 1, group_names[group_idx]);
            break;
         case 2:
            printf( "Single candidate found at %c%d:\n",
                     "abcdefghijklmnopqrst"[soln % su->grid_size],
                     soln / su->grid_size + 1);
            break;
         case 3:
            printf( "%d is limited to one line or column in %s:\n",
                  soln, group_names[group_idx]);
            break;
         case 4:
            printf( "%d exists as double pair in %s:\n", soln,
                        group_names[group_idx]);
            break;
         case 5:
            printf( "Naked pair found in %s:\n",
                        group_names[group_idx]);
            break;
         case 6:
            printf( "Naked triple found in %s:\n",
                        group_names[group_idx]);
            break;
         case 7:
            printf( "Naked quad found in %s:\n",
                        group_names[group_idx]);
            break;
         case 8:
            printf( "Hidden pair found in %s:\n",
                        group_names[group_idx]);
            break;
         case 9:
            printf( "Hidden triple found in %s:\n",
                        group_names[group_idx]);
            break;
         case 10:
            printf( "Hidden quad found in %s:\n",
                        group_names[group_idx]);
            break;
         case 11:
            printf( "X-Wings of %d found:\n", soln);
            break;
         case 12:
            printf( "Simplified guess at %c%d gave improvements:\n",
                     "abcdefghijklmnopqrst"[soln % su->grid_size],
                     soln / su->grid_size + 1);
            break;
         case 13:
            printf( "Double double found for %d\n", soln);
            break;
         case 14:
            printf( "Type II double double found for %d\n", soln);
            break;
         case 99:
            printf( "Guessed at %c%d:\n",
                     "abcdefghijklmnopqrst"[soln % su->grid_size],
                     soln / su->grid_size + 1);
            break;
         case PUZZLE_IS_UNSOLVABLE:
//          printf( "Unsolvable (2)\n");
            break;
         case MULTIPLE_SOLUTIONS_FOUND:
            printf( "Multiple solns\n");
            break;
         default:
            printf( "Unexpected value: %d\n", method);
            break;
         }
      show_sudoku_grid( su);
      for( i = n_cells_solved = 0; i < su->n_cells; i++)
         if( su->solutions[i])
            n_cells_solved++;
         else if( !su->possibles[i])
            soln = -99;
      step++;
      printf( "Step %d; %d cells found\n", step, n_cells_solved);
      }
   if( multiple_solutions)
      {
      printf( "Example solution:\n");
      show_sudoku_grid( an_example_solution);
      }
   else
      {
      if( soln == -99)
         printf( "UNSOLVABLE PUZZLE\n");
      printf( (n_cells_solved == su->n_cells) ? "Solution found" : "No solution found");
      }
   return( 0);
}
